/* datatypes.c */
/* This is part of the source for the Core Join Calculus Compiler cjcc */
/* Copyright (C) 1996 Peter Selinger */
/* This is free software under the terms of the GNU General Public License. */

#include <stdio.h>
#include <stdlib.h>
#include "declarations.h"

/* A semantic value can be: */

#define STRI 1  /* string */
#define LIST 2  /* list of sem. values */
#define NUM 3   /* integer */
#define RECEIVER 4   /* a receiver (see types.c) */
#define BASICDEF 5  /* a basicdef (see types.c) */ 
#define DEFAPAIR 6  /* a defa_pair (see types.c) */

typedef struct sem {
  int type;     /* one of STRI,LIST,NUM,etc */
  union {
    char *_c;
    struct list *_l;
    int _i;
    receivertype *_r;
    basicdeftype *_b;
    defa_pair *_d;
  } _semval;
} sem;
#define cct _semval._c   /* char content */
#define lct _semval._l   /* list content etc */
#define ict _semval._i
#define rct _semval._r
#define bct _semval._b
#define dct _semval._d

typedef struct list {
  sem *x;  /* list element */
  struct list *next;
} list;

list *emptylist(void) {
  return NULL;
}

int isempty(list *l) {
  if (l==NULL)
    return 1;
  return 0;
}

sem *head(list *l) {
  if (l!=NULL)
    return l->x;
  return NULL;
}

list *tail(list *l) {
  if (l!=NULL)
    return l->next;
  return NULL;
}

list *cons(sem *x, list *l) {
  list *p=(list *)calloc(1,sizeof(list));
  if (p!=NULL) {
    p->x=x;
    p->next=l;
  }
  return p;
}

list *append(list *l, list *m) {
  if (l==NULL)
    return m;
  if (l->next==NULL) {
    l->next=m;
    return l;
  }
  append(l->next,m);
  return l;
}

int member(sem *x, list *l) {
  if (l==NULL)
    return 0;
  if (semeq(x,l->x))
    return 1;
  return member(x,l->next);
}

list *setify(list *l) {
  if (l==NULL)
    return NULL;
  if (member(l->x,l->next))
    return setify(l->next);
  l->next=setify(l->next);
  return l;
}

void listprint(FILE *file,list *l) {
  if (l==NULL)
    return;
  semprint(file,l->x);
  listprint(file,l->next);
}

void longlistprint(FILE *file, list *l) {
  fprintf(file,"[");
  while (l!=NULL) {
    longsemprint(file,l->x);
    l=l->next;
  }
  fprintf(file,"]");
}

int listeq(list *l, list *m) {
  if (l==NULL && m==NULL)
    return 1;
  if (l!=NULL && m!=NULL)
    return (semeq(l->x,m->x) && listeq(l->next,m->next));
  return 0;
}

int disjoint(list *l, list *m) {
  while (l!=NULL) {
    if (member(l->x,m))
      return 0;
    l=l->next;
  }
  return 1;
}

list *singleton(sem *s) {
  return cons(s, NULL);
}

list *doubleton(sem *s, sem *t) {
  return cons(s, cons(t, NULL));
}

list *listclone(list *l) {
  if (l==NULL)
    return NULL;
  return cons(l->x,listclone(l->next));
}

list *minus(list *l, list *m) {
  if (l==NULL)
    return NULL;
  if (member(l->x,m))
    return minus(l->next,m);
  return cons(l->x,minus(l->next,m));
}

int length(list *l) {
  if (l==NULL)
    return 0;
  return 1+length(l->next);
}

sem *listelement(list *l, int i) {
  if (l==NULL)
    return NULL;
  if (i<=0)
    return (l->x);
  else
    return listelement(l->next, i-1);
}

int semeq(sem *a, sem *b) {
  if (a->type==STRI && b->type==STRI)
    return streq(a->cct,b->cct);
  if (a->type==LIST && b->type==LIST) 
    return listeq(a->lct,b->lct);
  if (a->type==NUM && b->type==NUM)
    return a->ict==b->ict;
  if (a->type==DEFAPAIR && b->type==DEFAPAIR)
    return streq(a->dct->d,b->dct->d) && a->dct->a==b->dct->a;
  return 0;
}

int streq(char *c, char *d) {
  while (*c!='\0' && *d!='\0') {
    if (*c!=*d)
      return 0;
    c++;
    d++;
  }
  if (*d=='\0' && *c=='\0') 
    return 1;
  return 0;
}

void semprint(FILE *file, sem *a) {
  if (a==NULL) {
    fprintf(file,"NULL");
    return;
  }
  if (a->type==STRI)
    fprintf(file, "%s", a->cct); 
  else if (a->type==LIST) 
    listprint(file,a->lct);
  else if (a->type==NUM)
    fprintf(file,"%d", a->ict);
  else if (a->type==DEFAPAIR)
    fprintf(file,"<%s,%d>",a->dct->d,a->dct->a);
}

void longsemprint(FILE *file,sem *a) {
  if (a==NULL) {
    fprintf(file,"NULL ");
    return;
  }
  if (a->type==STRI) 
    fprintf(file,"%s ", a->cct); 
  else if (a->type==LIST)  
    longlistprint(file,a->lct);
  else if (a->type==NUM)
    fprintf(file,"%d ", a->ict);
  else if (a->type==DEFAPAIR)
    fprintf(file,"<%s,%d>",a->dct->d,a->dct->a);
  else
    fprintf(file,"???? ");
}

sem *instr(char *a) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  if (p!=NULL) { 
    p->type=STRI;
    p->cct=a;
  }
  return p;
}

sem *inlist(list *l) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  if (p!=NULL) { 
    p->type=LIST;
    p->lct=l;
  }
  return p;
}
 
sem *innum(int n) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  if (p!=NULL) { 
    p->type=NUM;
    p->ict=n;
  }
  return p;
}

sem *inreceiver(receivertype r) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  receivertype *rcontainer=(receivertype *)calloc(1,sizeof(receivertype));
  if (p!=NULL && rcontainer!=NULL) { 
    p->type=RECEIVER;
    p->rct=rcontainer;
    *rcontainer=r;
  }
  return p;
}
 
sem *indefa(defa_pair r) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  defa_pair *rcontainer=(defa_pair *)calloc(1,sizeof(defa_pair));
  if (p!=NULL && rcontainer!=NULL) { 
    p->type=DEFAPAIR;
    p->dct=rcontainer;
    *rcontainer=r;
  }
  return p;
}
 
sem *inbasicdef(basicdeftype r) {
  sem *p=(sem *)calloc(1,sizeof(sem));
  basicdeftype *rcontainer=(basicdeftype *)calloc(1,sizeof(basicdeftype));
  if (p!=NULL && rcontainer!=NULL) { 
    p->type=BASICDEF;
    p->bct=rcontainer;
    *rcontainer=r;
  }
  return p;
}
 
char *strclone(char *s) {
  int i;
  char *t;
  
  for (i=0; s[i]!='\0'; i++);
  if ((t=(char *) calloc(i+1,sizeof(char)))!=NULL) {
    for (i=0; s[i]!='\0'; i++) {
      t[i]=s[i];
    }
    t[i]='\0';
  }
  return t;
}

sem *app(sem *s,sem *t) {
  return inlist(cons(s, cons(t, NULL)));
}

sem *emptysem(void) {
  return inlist(NULL);
}

/* Two lists of pairs s=[(d,a),...] and t=[(d,a),...] are consistent
   if in their union, for every d there is at most one a. We assume
   without further checking that s and t and individually consistent,
   and that s and t are in fact lists of pairs.  d, a1 and a2 are for
   diagnosis: they return the values where a violation occurred. */

int consistent(list *s, list *t, char **d, int *a1, int *a2) { 
  while (s!=NULL) { 
    if (!consis(s->x->dct,t,d,a1,a2)) 
      return 0; 
    s=s->next; 
  } 
  return 1; 
}

int consis(defa_pair *q, list *t, char **d, int *a1, int *a2) {
  while (t!=NULL) {
    if (streq(*d = t->x->dct->d , q->d)
	&& (*a1 = t->x->dct->a) != (*a2 = q->a))       
      return 0;
    t=t->next;
  }
  return 1;
}

char *removequotes(char *s) {
/* removes the leading and final quotation mark from a string */
  char *t;
  
  for (t=s+1; *t!='\"' && *t!='\0'; t++);
  *t='\0';
  return s+1;
}
